# PYTHON & PASCO: Luftmotstand med muffinsformer i fritt fall (bestfit)
# Versjon: 15.6.2021
# Kontakt: koding@astronomen.no
from pylab import *
import pandas as pd

# -----------------------Rediger fra her--------------------------------------
# Filnavnet til dataene
filnavn = "Muffinsformer.csv"

# Tilpassing av dataene til området vi ønsker å analysere
startpunkt_data = 11    # Hvor langt ut i datasettet vi setter første punkt
sluttpunkt_data = 38    # Hvor langt ut i datasettet vi setter siste punkt

# Konstanter til bruk i modell
m = 0.30*10**(-3)                   # Masse, kg
g = 9.81                            # Tyngdeakselerasjon, m/s^2
dt = 0.0001                         # Tidssteget som brukes i simuleringen
k_verdier = linspace(0,0.002,100)   # k-verdiene vi skal utforske

# Akselerasjonsfunksjon
def a(v,k):
    # Utregning av krefter
    G = m*g             # Tyngdekraft, N
    L = k*v**2          # Luftmotstand, N

    # Slippes formen fra under sensor skal sum_F = G - L
    # Slippes formen fra over sensor skal sum_F = L - G
    sum_F = L - G       # Kraftsum, N
    aks = sum_F/m       # Akselerasjon, m/s^2
    return aks
# ----------------------Ikke rediger under------------------------------------

# Simulering
def simulering(s,v,t,a):
    s_sim = [s]; v_sim = [v]; t_sim = [t]  #Lister for lagring av simulering

    # Kalkulerer neste verdier og lagrer dem i lister
    while t < t_slutt:
        v = v + a(v,k)*dt   # Regner ut neste fart, overskriver gammel
        s = s + v*dt        # Regner ut neste posisjon, overskriver gammel
        t = t + dt          # Regner ut neste tidspunkt

        v_sim.append(v)     # Lagrer den nye farten i liste
        s_sim.append(s)     # Lagrer den nye posisjonen i liste
        t_sim.append(t)     # Lagrer den nye tiden i liste

    return t_sim, s_sim         # Returnerer simulerte verdier

# Funksjon for å finne sum av differanser mellom modell og data
def differanse(t_sim, s_sim, t_data, s_data):
    i = 0                           # Index som løper gjennom simuleringen
    sum_differanse = 0              # Variabel for lagring av differanse
    for j in range(len(t_data)):    # Løper gjennom dataene

        # Oppdaterer indeksen så lenge det er mulig å komme nærmere måledata.
        while abs(t_sim[i+1] - t_data[j]) <  abs(t_sim[i] - t_data[j]):
            i+=1
        sum_differanse += abs(s_sim[i]-s_data[j]) # Kumulativ differanse
        i+=1                                # Sørger for at man kan lete videre
    return sum_differanse                   # Returnerer kumulativ differanse

# Importering av data
csv_data = pd.read_csv(open(filnavn, 'r'), delimiter=";", decimal=",")
t_data = csv_data.iloc[:,1].to_numpy()
s_data = csv_data.iloc[:,2].to_numpy()
t_data = t_data[~isnan(t_data)]
s_data = s_data[~isnan(s_data)]

# Tilpassing av datasett utifra startpunkt og sluttpunkt
t_data = t_data[startpunkt_data:sluttpunkt_data]
t_data = t_data - t_data[0]
s_data = s_data[startpunkt_data:sluttpunkt_data]

# Finne den beste verdien for k
t_slutt = t_data[-1] + dt               # Sluttid for simulering
score = 100                             # Startscore som garanterer overskriving
for k in k_verdier:                     # Verdier for k å iterere over
    s = s_data[0]                       # Startposisjon for simulering
    v = 0                               # Starfart for simulering
    t = 0                               # Starttidspunkt for simulering
    t_sim, s_sim = simulering(s,v,t,a)  # Lagrer simulering
    score_temp = differanse(t_sim, s_sim, t_data, s_data) # Kalkulerer differanse
    if score_temp < score:          # Hvis vi har bedre score, lagres resultatet
        score = score_temp              # Overskriver score
        best_k = k                      # Lagrer beste verdi for k
        best_s_sim = s_sim              # Lagrer den beste simuleringen

print('')
print('Beste verdi for k er:',
      round(best_k,5))
print('Gjennomsnittelig differanse per punkt: ',
         round(score/len(t_data),4))
# Plot av data mot modell
plot(t_data,s_data,'gx')
plot(t_sim,best_s_sim)
xlabel('Tid / s')
ylabel('Posisjon / m')
title('Posisjonsgraf for muffinsform')
legend(['Måledata','Modell'])
show()
